use fun::input::Joystick;
use fun::math::Vector;
use fun::object::Object;
use fun::object::Transform;
use fun::render::Color;
use fun::render::MaterialName;
use fun::render::Mesh;
use fun::shapes;
use side::Side;
use side::Side::*;

const MIN_HEIGHT: f32 = 0.0;
const MAX_HEIGHT: f32 = 2.0;

pub struct Flag {
    side: Side,
    transform: Transform,
    flag: Mesh,
    pole: Mesh,
    time: f32,
    dead: bool,
}

impl Flag {
    pub fn new(side: Side) -> Box<Object> {
        Box::new(Flag {
            side: side,
            transform: {
                use fun::math::Rotation;
                use fun::math::Vector;

                let mut transform = Transform::IDENTITY;

                transform.orientation.position = match side {
                    Left => Vector::new(-2.0, 0.0, 5.0),
                    Right => Vector::new(2.0, 0.0, 5.0),
                };

                transform.orientation.rotation = match side {
                    Left => Rotation::new(0.0, 0.75, 0.0),
                    Right => Rotation::new(0.0, 0.25, 0.0),
                };

                transform
            },
            flag: {
                use fun::math::Vector;

                const WIDTH: u16 = 15;
                const HEIGHT: u16 = 15;
                const TOTAL_VERTICES: u16 = WIDTH * HEIGHT;
                const TOTAL_INDICES: u16 = 6 * TOTAL_VERTICES;

                let mut vertices = Vec::with_capacity(TOTAL_VERTICES as usize);
                let mut indices = Vec::with_capacity(TOTAL_INDICES as usize);

                for y in 0..HEIGHT {
                    for x in 0..WIDTH {
                        let vertex = Vector::new(0.0, x as f32, y as f32);

                        vertices.push(vertex);

                        let index = x + y * HEIGHT;
                        let up = x + (y + 1) * HEIGHT;
                        let right = x + 1 + y * HEIGHT;
                        let up_right = x + 1 + (y + 1) * HEIGHT;

                        if x < WIDTH - 1 && y < HEIGHT - 1 {
                            indices.push(index);
                            indices.push(up);
                            indices.push(right);

                            indices.push(up_right);
                            indices.push(right);
                            indices.push(up);
                        }
                    }
                }

                let normals = vertices.clone();

                Mesh {
                    vertices: vertices,
                    normals: normals,
                    indices: indices,
                    color: {
                        use fun::render::Color;

                        match side {
                            Left => Color::BLUE,
                            Right => Color::RED,
                        }
                    },
                    skew: 0.1 * Vector::ONE,
                    material: MaterialName::Wireframe,
                }
            },
            pole: {
                let color = Color::WHITE;
                let skew = Vector::new(0.1, 0.0, 0.1);

                shapes::hexahedron(MaterialName::Wireframe, color, skew)
            },
            time: 0.0,
            dead: false,
        })
    }

    fn pole_height(&self) -> f32 {
        let mut height = match self.side {
            Left => unsafe { ::LEFT_FLAG_HEIGHT },
            Right => unsafe { ::RIGHT_FLAG_HEIGHT },
        };

        if height > MAX_HEIGHT {
            height = MAX_HEIGHT;
        }

        if height < MIN_HEIGHT {
            height = MIN_HEIGHT;
        }

        height
    }
}

impl Object for Flag {
    fn transform_ref(&self) -> &Transform {
        &self.transform
    }

    fn transform_mut(&mut self) -> &mut Transform {
        &mut self.transform
    }

    fn meshes(&self) -> Vec<&Mesh> {
        vec![&self.flag, &self.pole]
    }

    fn add(&self) -> Vec<Box<Object>> {
        if self.dead {
            use super::Timber;
            use super::Football;

            let height = self.pole_height();
            let timber = Timber::new(self.side, height);
            let football = Football::new(self.side, height);

            vec![timber, football]
        } else {
            Vec::with_capacity(0)
        }
    }

    fn remove(&self) -> bool {
        self.dead
    }

    fn update(&mut self, _: &(Joystick, Joystick)) {
        for (i, vector) in self.flag.vertices.iter_mut().enumerate() {
            vector.x = (i as f32 * self.time / 2_000.0).sin();
        }

        let height = Self::pole_height(self);
        let mut flag_height = height / 15.0;

        if flag_height > 0.1 {
            flag_height = 0.1;
        }

        self.flag.skew.z = flag_height;
        self.flag.skew.y = flag_height;
        self.pole.skew.y = height;
        self.transform.orientation.position.y = height;

        self.time = {
            use std::f32;

            let diff = f32::MAX - self.time;
            let delta = unsafe { ::fun::global::TIMER.delta };

            if diff > delta {
                self.time + delta
            } else {
                delta - diff
            }
        };

        match self.side {
            Left => {
                self.dead = unsafe { ::LEFT_KICK };
            }
            Right => {
                self.dead = unsafe { ::RIGHT_KICK };
            }
        }
    }
}
