use fun::input::Joystick;
use fun::math::Vector;
use fun::object::Object;
use fun::object::Transform;
use fun::render::Color;
use fun::render::MaterialName;
use fun::render::Mesh;
use fun::shapes;
use side::Side;
use side::Side::*;

const GRAVITY: f32 = -0.0005;
const TERMINAL_VELOCITY: f32 = -0.1;

pub struct Football {
    side: Side,
    transform: Transform,
    mesh: Mesh,
    velocity: f32,
    dead: bool,
}

impl Football {
    pub fn new(side: Side, height: f32) -> Box<Object> {
        Box::new(Football {
            side: side,
            transform: {
                let mut transform = Transform::IDENTITY;

                let x = match side {
                    Left => -2.0,
                    Right => 2.0,
                };

                transform.orientation.position = Vector::new(x, 2.0 * height, 5.0);

                transform
            },
            mesh: {
                let color = Color::YELLOW;
                let skew = Vector::new(0.5, 0.25, 0.2);

                shapes::octahedron(MaterialName::Wireframe, color, skew)
            },
            velocity: 0.0,
            dead: false,
        })
    }
}

impl Object for Football {
    fn transform_ref(&self) -> &Transform {
        &self.transform
    }

    fn transform_mut(&mut self) -> &mut Transform {
        &mut self.transform
    }

    fn meshes(&self) -> Vec<&Mesh> {
        vec![&self.mesh]
    }

    fn remove(&self) -> bool {
        self.dead
    }

    fn update(&mut self, _: &(Joystick, Joystick)) {
        let delta = unsafe { &::fun::global::TIMER }.delta;

        self.transform.orientation.position.x += delta * match self.side {
            Left => 0.07,
            Right => -0.07,
        };

        self.transform.orientation.position.y += delta * self.velocity;

        self.velocity += delta * GRAVITY;

        if self.velocity < TERMINAL_VELOCITY {
            self.velocity = TERMINAL_VELOCITY;
        }

        if self.transform.orientation.position.y < 0.0 {
            self.dead = true;
        }

        self.transform.orientation.rotation.x += delta * 0.01;
        self.transform.orientation.rotation.z += delta * 0.001;

        let position = self.transform.orientation.position;
        let position_2d = (position.x, position.y);

        use std::f32;

        match self.side {
            Left => unsafe {
                if ::LEFT_FOOTBALL_POSITION == (f32::INFINITY, f32::INFINITY) {
                    self.dead = true;
                }
                
                ::LEFT_FOOTBALL_POSITION = position_2d;
            },
            Right => unsafe {
                if ::RIGHT_FOOTBALL_POSITION == (f32::INFINITY, f32::INFINITY) {
                    self.dead = true;
                }

                ::RIGHT_FOOTBALL_POSITION = position_2d;
            },
        }
    }
}
