use fun::input::Joystick;
use fun::object::Object;
use fun::object::Transform;
use fun::render::Color;
use fun::render::MaterialName;
use fun::render::Mesh;

pub struct Ground {
    transform: Transform,
    mesh: Mesh,
    shake_mesh: Mesh,
    time: f32,
}

impl Ground {
    pub fn new() -> Box<Object> {
        let mesh = {
            use fun::math::Vector;

            const WIDTH: u16 = 15;
            const HEIGHT: u16 = 15;
            const TOTAL_VERTICES: u16 = WIDTH * HEIGHT;
            const TOTAL_INDICES: u16 = 6 * TOTAL_VERTICES;

            let mut vertices = Vec::with_capacity(TOTAL_VERTICES as usize);
            let mut indices = Vec::with_capacity(TOTAL_INDICES as usize);

            for y in 0..HEIGHT {
                for x in 0..WIDTH {
                    let vertex = Vector::new(x as f32, 0.0, y as f32);

                    vertices.push(vertex);

                    let index = x + y * HEIGHT;
                    let up = x + (y + 1) * HEIGHT;
                    let right = x + 1 + y * HEIGHT;
                    let up_right = x + 1 + (y + 1) * HEIGHT;

                    if x < WIDTH - 1 && y < HEIGHT - 1 {
                        indices.push(index);
                        indices.push(up);
                        indices.push(right);

                        indices.push(up_right);
                        indices.push(right);
                        indices.push(up);
                    }
                }
            }

            let normals = vertices.clone();

            Mesh {
                vertices: vertices,
                normals: normals,
                indices: indices,
                color: Color::WHITE,
                skew: 1.0 * Vector::ONE,
                material: MaterialName::Wireframe,
            }
        };
        Box::new(Ground {
            transform: {
                use fun::math::Vector;

                let mut transform = Transform::IDENTITY;

                transform.orientation.position = Vector::new(-5.0, 0.0, 0.0);

                transform
            },
            mesh: mesh.clone(),
            shake_mesh: mesh,
            time: 0.0,
        })
    }

    fn shake() -> bool {
        unsafe { ::LEFT_SHAKE_GROUND || ::RIGHT_SHAKE_GROUND }
    }
}

impl Object for Ground {
    fn transform_ref(&self) -> &Transform {
        &self.transform
    }

    fn transform_mut(&mut self) -> &mut Transform {
        &mut self.transform
    }

    fn meshes(&self) -> Vec<&Mesh> {
        if Ground::shake() {
            vec![&self.shake_mesh]
        } else {
            vec![&self.mesh]
        }
    }

    fn update(&mut self, _: &(Joystick, Joystick)) {
        if Ground::shake() {
            for (i, vector) in self.shake_mesh.vertices.iter_mut().enumerate() {
                let mut y = (i as f32 * self.time).tan() / 1_000.0;

                const MIN: f32 = 0.0;
                const MAX: f32 = 0.5;

                if y > MAX {
                    y = MAX;
                }

                if y < MIN {
                    y = MIN;
                }

                vector.y = y;
            }

            self.time = {
                use std::f32;

                let diff = f32::MAX - self.time;
                let delta = unsafe { ::fun::global::TIMER.delta };

                if diff > delta {
                    self.time + delta
                } else {
                    delta - diff
                }
            };
        }
    }
}
