use self::State::*;
use fun::input::Joystick;
use fun::math::Vector;
use fun::object::Object;
use fun::object::Transform;
use fun::render::Mesh;
use fun::shapes;
use side::Side;
use side::Side::*;

const GRAVITY: Velocity = -0.001;
const TERMINAL_VELOCITY: Velocity = -0.1;

const SALUTE_HEIGHT: f32 = 1.0;
const PRAY_HEIGHT: f32 = 0.4;
const STUN_HEIGHT: f32 = 1.0;

const KICK_OUCH_TIME: Time = 30.0;
const TOTAL_KICK_TIME: Time = 60.0;

type Altitude = f32;
type Velocity = f32;
type Time = f32;
type GainWeight = bool;
type Success = bool;

#[derive(Clone, Copy, PartialEq)]
enum State {
    Salute,
    Crouch,
    Jump(Altitude, Velocity, GainWeight),
    Land(Time),
    Stun(Time),
    Kick(Time),
    Mitt(Time, Success),
    Throw(Time, Altitude),
}

impl State {
    fn next(self, side: Side, joystick: Joystick) -> Self {
        use fun::input::Joystick::*;

        let delta = unsafe { &::fun::global::TIMER }.delta;

        let stun = match side {
            Side::Left => unsafe { ::LEFT_SHAKE },
            Side::Right => unsafe { ::RIGHT_SHAKE },
        };

        match side {
            Side::Left => unsafe {
                ::LEFT_KICK = false;
                ::LEFT_THREW_BALL = false;
            },
            Side::Right => unsafe {
                ::RIGHT_KICK = false;
                ::RIGHT_THREW_BALL = false;
            },
        }

        match self {
            Salute => {
                if stun {
                    Stun(0.0)
                } else {
                    match joystick {
                        Up | UpLeft | UpRight => Jump(0.1, 0.05, false),
                        Down | DownLeft | DownRight => Crouch,
                        Left => Mitt(0.0, false),
                        Right => {
                            match side {
                                Side::Left => {
                                    if unsafe { ::LEFT_HAS_BALL } {
                                        Throw(0.0, 0.0)
                                    } else {
                                        Kick(0.0)
                                    }
                                }
                                Side::Right => {
                                    if unsafe { ::RIGHT_HAS_BALL } {
                                        Throw(0.0, 0.0)
                                    } else {
                                        Kick(0.0)
                                    }
                                }
                            }
                        },
                        _ => Salute,
                    }
                }
            },
            Crouch => {
                if stun {
                    Stun(0.0)
                } else {
                    match joystick {
                        Down | DownLeft | DownRight => Crouch,
                        _ => Salute,
                    }
                }
            },
            Jump(altitude, mut velocity, mut gain_weight) => {
                if !gain_weight && velocity < 0.0 {
                    match joystick {
                        Down | DownLeft | DownRight => {
                            gain_weight = true;
                            velocity = TERMINAL_VELOCITY;
                        },
                        Right => {
                            match side {
                                Side::Left => {
                                    if unsafe { ::LEFT_HAS_BALL } {
                                        return Throw(0.0, altitude)
                                    }
                                }
                                Side::Right => {
                                    if unsafe { ::RIGHT_HAS_BALL } {
                                        return Throw(0.0, altitude)
                                    }
                                }
                            }
                        },
                        _ => (),
                    }
                }
                let next_altitude = altitude + velocity * delta;

                if next_altitude <= 0.0 {
                    return Land(0.0)
                }

                let mut next_velocity = velocity + GRAVITY * delta;

                if next_velocity <= TERMINAL_VELOCITY {
                    next_velocity = TERMINAL_VELOCITY;
                }

                Jump(next_altitude, next_velocity, gain_weight)
            },
            Land(time) => {
                match side {
                    Side::Left => {
                        unsafe {
                            ::RIGHT_SHAKE = time == 0.0;
                            ::RIGHT_SHAKE_GROUND = true;
                        }
                    },
                    Side::Right => {
                        unsafe {
                            ::LEFT_SHAKE = time == 0.0;
                            ::LEFT_SHAKE_GROUND = true;
                        }
                    },
                }

                let delta = unsafe { &::fun::global::TIMER }.delta;

                const TOTAL_TIME: Time = 30.0;

                let next_time = time + delta;

                if next_time > TOTAL_TIME {
                    match side {
                        Side::Left => {
                            unsafe {
                                ::RIGHT_SHAKE_GROUND = false;
                            }
                        },
                        Side::Right => {
                            unsafe {
                                ::LEFT_SHAKE_GROUND = false;
                            }
                        },
                    }
                    Salute
                } else {
                    Land(next_time)
                }
            },
            Stun(time) => {
                let delta = unsafe { &::fun::global::TIMER }.delta;

                const TOTAL_TIME: Time = 200.0;

                let next_time = time + delta;

                if next_time > TOTAL_TIME {
                    Salute
                } else {
                    Stun(next_time)
                }
            },
            Kick(time) => {
                let delta = unsafe { &::fun::global::TIMER }.delta;

                let next_time = time + delta;

                if next_time > KICK_OUCH_TIME {
                    match side {
                        Side::Left => unsafe {
                            ::LEFT_KICK = true;
                        },
                        Side::Right => unsafe {
                            ::RIGHT_KICK = true;
                        },
                    }
                }

                if next_time > TOTAL_KICK_TIME {
                    Salute
                } else {
                    Kick(next_time)
                }
            }
            Mitt(time, mut success) => {
                let delta = unsafe { &::fun::global::TIMER }.delta;

                if !success {
                    let football_position = match side {
                        Side::Left => unsafe {
                            ::RIGHT_FOOTBALL_POSITION
                        },
                        Side::Right => unsafe {
                            ::LEFT_FOOTBALL_POSITION
                        },
                    };

                    let x = football_position.0;
                    let y = football_position.1;

                    let x_overlap = match side {
                        Side::Left => x < -1.5 && x > -2.5,
                        Side::Right => x > 1.5 && x < 2.5,
                    };
                    let scale = 2.0 * SALUTE_HEIGHT;

                    let y_overlap = y > 0.8 * scale && y < 1.2 * scale;

                    success = x_overlap && y_overlap;

                    if success {
                        use std::f32;

                        match side {
                            Side::Left => unsafe {
                                ::LEFT_HAS_BALL = true;
                                ::RIGHT_FOOTBALL_POSITION = (f32::INFINITY, f32::INFINITY);
                            },
                            Side::Right => unsafe {
                                ::RIGHT_HAS_BALL = true;
                                ::LEFT_FOOTBALL_POSITION = (f32::INFINITY, f32::INFINITY);
                            },
                        }
                    }
                }

                const TOTAL_TIME: Time = 90.0;

                let next_time = time + delta;

                if next_time > TOTAL_TIME {
                    Salute
                } else {
                    Mitt(next_time, success)
                }
            },
            Throw(time, altitude) => {
                let delta = unsafe { &::fun::global::TIMER }.delta;

                const TOTAL_TIME: Time = 15.0;

                let next_time = time + delta;

                if next_time > TOTAL_TIME {
                    match side {
                        Side::Left => unsafe {
                            ::LEFT_THREW_BALL = true;
                            ::LEFT_HAS_BALL = false;
                        },
                        Side::Right => unsafe {
                            ::RIGHT_THREW_BALL = true;
                            ::RIGHT_HAS_BALL = false;
                        },
                    }

                    if altitude > 0.0 {
                        Jump(altitude, 0.0, false)
                    } else {
                        Salute
                    }
                } else {
                    Throw(next_time, altitude)
                }
            },
        }
    }

    fn altitude(self) -> Altitude {
        match self {
            Salute => SALUTE_HEIGHT,
            Crouch => PRAY_HEIGHT,
            Jump(altitude, _, _) => SALUTE_HEIGHT + altitude,
            Land(_) => PRAY_HEIGHT,
            Stun(_) => STUN_HEIGHT,
            Kick(_) => SALUTE_HEIGHT,
            Mitt(_, _) => SALUTE_HEIGHT,
            Throw(_, altitude) => SALUTE_HEIGHT + altitude,
        }
    }
}

pub struct Patriot {
    time: Time,
    side: Side,
    state: State,
    transform: Transform,
    salute_mesh: Mesh,
    crouch_mesh: Mesh,
    stun_mesh: Mesh,
    mitt_mesh: Mesh,
    dead: bool,
}

impl Patriot {
    pub fn new(side: Side) -> Box<Object> {
        Box::new(Patriot {
            time: 0.0,
            side: side,
            state: Salute,
            transform: {
                use fun::math::Rotation;

                let mut transform = Transform::IDENTITY;

                let y = Salute.altitude();

                transform.orientation.position = match side {
                    Left => Vector::new(-3.0, y, 5.0),
                    Right => Vector::new(3.0, y, 5.0),
                };

                transform.orientation.rotation = match side {
                    Left => Rotation::IDENTITY,
                    Right => Rotation::IDENTITY,
                };

                transform
            },
            salute_mesh: Self::salute_mesh(side),
            crouch_mesh: Self::crouch_mesh(side),
            stun_mesh: Self::stun_mesh(side),
            mitt_mesh: Self::mitt_mesh(side),
            dead: false,
        })
    }

    fn salute_mesh(side: Side) -> Mesh {
        use fun::render::Color;
        use fun::render::MaterialName;

        let color = match side {
            Left => Color::RED,
            Right => Color::BLUE,
        };
        let skew = Vector::new(0.2, SALUTE_HEIGHT, 0.1);

        shapes::hexahedron(MaterialName::Wireframe, color, skew)
    }

    fn crouch_mesh(side: Side) -> Mesh {
        use fun::render::Color;
        use fun::render::MaterialName;

        let color = match side {
            Left => Color::RED,
            Right => Color::BLUE,
        };
        let skew = Vector::new(0.2, PRAY_HEIGHT, 0.1);

        shapes::hexahedron(MaterialName::Wireframe, color, skew)
    }

    fn stun_mesh(side: Side) -> Mesh {
        use fun::render::Color;
        use fun::render::MaterialName;

        let color = match side {
            Left => Color::RED,
            Right => Color::BLUE,
        };
        let skew = Vector::new(0.2, STUN_HEIGHT, 0.1);

        shapes::hexahedron(MaterialName::Wireframe, color, skew)
    }

    fn mitt_mesh(side: Side) -> Mesh {
        use fun::render::Color;
        use fun::render::MaterialName;

        let color = match side {
            Left => Color::RED,
            Right => Color::BLUE,
        };
        let x = match side {
            Left => 0.5,
            Right => -0.5,
        };
        let skew = Vector::ONE * 0.4;

        let mut mesh = shapes::tetrahedron(MaterialName::Wireframe, color, skew);

        for i in 0..mesh.vertices.len() {
            mesh.vertices[i] += Vector::new(2.0 * x, 2.0, 0.0);
        }

        mesh
    }
}

impl Object for Patriot {
    fn transform_ref(&self) -> &Transform {
        &self.transform
    }

    fn transform_mut(&mut self) -> &mut Transform {
        &mut self.transform
    }

    fn meshes(&self) -> Vec<&Mesh> {
        match self.state {
            Salute => vec![&self.salute_mesh],
            Crouch => vec![&self.crouch_mesh],
            Jump(_, _, _) => vec![&self.salute_mesh],
            Land(_) => vec![&self.crouch_mesh],
            Stun(_) => vec![&self.stun_mesh],
            Kick(_) => vec![&self.salute_mesh],
            Mitt(_, _) => vec![&self.salute_mesh, &self.mitt_mesh],
            Throw(_, _) => vec![&self.salute_mesh],
        }
    }
    fn add(&self) -> Vec<Box<Object>> {
        if self.dead {
            use super::Firework;
            use fun::math::Rotation;
            use fun::math::Vector;
            use fun::physics::Motion;
            use fun::render::MaterialName;

            let mut objects = Vec::default();

            for mesh in self.meshes() {
                let vertices = mesh.vertices.clone();
                let indices = mesh.indices.clone();

                for i in 0..indices.len() / 3 {
                    let index_a = indices[i * 3] as usize;
                    let index_b = indices[i * 3 + 1] as usize;
                    let index_c = indices[i * 3 + 2] as usize;

                    let vertex_a = vertices[index_a];
                    let vertex_b = vertices[index_b];
                    let vertex_c = vertices[index_c];

                    let mesh = Mesh {
                        vertices: vec![vertex_a, vertex_b, vertex_c],
                        normals: vec![vertex_a, vertex_b, vertex_c],
                        indices: vec![
                            0, 1, 2,
                            2, 1, 0,
                        ],
                        color: mesh.color,
                        skew: mesh.skew,
                        material: MaterialName::Wireframe,
                    };

                    let mut transform = self.transform.clone();

                    transform.motion = Motion {
                        ballistic: true,
                        velocity:{
                            let x = (i as f32).sin();
                            let y = (i as f32).cos();
                            let z = (i as f32).tan();

                            Vector::new(x, y, z) / 10.0
                        },
                        angular_velocity: Rotation::IDENTITY,
                        force: Vector::ZERO,
                        torque: Vector::ZERO,
                    };

                    objects.push(Firework::new(transform, mesh));
                }
            }

            objects
        } else {
            use super::Football;

            let mut altitude = self.state.altitude();

            if let Jump(a, _, _) = self.state {
                altitude = a;
            }

            if self.side == Side::Left && unsafe { ::LEFT_THREW_BALL } {
                unsafe {
                    ::LEFT_THREW_BALL = false;
                }

                let football = Football::new(self.side, altitude);

                vec![football]
            } else if self.side == Side::Right && unsafe { ::RIGHT_THREW_BALL } {
                unsafe {
                    ::RIGHT_THREW_BALL = false;
                }

                let football = Football::new(self.side, altitude);

                vec![football]
            } else {
                Vec::with_capacity(0)
            }
        }
    }

    fn remove(&self) -> bool {
        self.dead
    }

    fn update(&mut self, players: &(Joystick, Joystick)) {
        use fun::math::Rotation;

        let joystick = match self.side {
            Left => players.0,
            Right => players.1,
        };

        self.state = self.state.next(self.side, joystick);
        self.transform.orientation.position.y = self.state.altitude();

        let crouch = self.state == Crouch;

        match self.side {
            Left => {
                unsafe {
                    ::LEFT_PRAY = crouch
                }
            },
            Right => {
                unsafe {
                    ::RIGHT_PRAY = crouch;
                }
            },
        }

        match self.state {
            Salute => {
                let growth = 0.002 * unsafe { &::fun::global::TIMER }.delta;

                match self.side {
                    Left => {
                        unsafe {
                            ::LEFT_FLAG_HEIGHT += growth;
                        }
                    },
                    Right => {
                        unsafe {
                            ::RIGHT_FLAG_HEIGHT += growth;
                        }
                    },
                }
            },
            Stun(_) => {
                self.time += unsafe { &::fun::global::TIMER }.delta;

                for (i, vector) in self.stun_mesh.vertices.iter_mut().enumerate() {
                    vector.z = (i as f32 * self.time).sin();
                }
            },
            Kick(time) => {
                let delta = unsafe { &::fun::global::TIMER }.delta;

                if time + delta >= TOTAL_KICK_TIME {
                    self.transform.orientation.rotation = Rotation::IDENTITY;
                } else {
                    let z = if time + delta > KICK_OUCH_TIME {
                        match self.side {
                            Left => self.transform.orientation.rotation.z + delta * 0.005,
                            Right => self.transform.orientation.rotation.z - delta * 0.005,
                        }
                    } else {
                        match self.side {
                            Left => self.transform.orientation.rotation.z - delta * 0.005,
                            Right => self.transform.orientation.rotation.z + delta * 0.005,
                        }
                    };

                    self.transform.orientation.rotation = Rotation::new(0.0, 0.0, z);
                }
            },
            _ => (),
        }

        {   // dead
            let football_position = match self.side {
                Left => unsafe {
                    ::RIGHT_FOOTBALL_POSITION
                },
                Right => unsafe {
                    ::LEFT_FOOTBALL_POSITION
                },
            };

            let overlap_x = (self.transform.orientation.position.x - football_position.0).abs() < 0.1;

            let y = football_position.1;

            let overlap_y = match self.state {
                Salute => y < 2.0 * SALUTE_HEIGHT,
                Crouch => y < 2.0 * PRAY_HEIGHT,
                Jump(altitude, _, _) => y < altitude + 2.0 * SALUTE_HEIGHT && y > altitude,
                Land(_) => y < 2.0 * PRAY_HEIGHT,
                Stun(_) => y < 2.0 * STUN_HEIGHT,
                Kick(_) => y < 2.0 * SALUTE_HEIGHT,
                Mitt(_, _) => y < 2.0 * SALUTE_HEIGHT,
                Throw(_, altitude) => y < altitude + 2.0 * SALUTE_HEIGHT && y > altitude,
            };

            self.dead = overlap_x && overlap_y
        }
    }
}
