use fun::input::Joystick;
use fun::math::Rotation;
use fun::math::Vector;
use fun::object::Object;
use fun::object::Transform;
use fun::render::Color;
use fun::render::MaterialName;
use fun::render::Mesh;
use side::Side;
use side::Side::*;

pub struct Timber {
    side: Side,
    transform: Transform,
    mesh: Mesh,
    dead: bool,
    height: f32,
}

impl Timber {
    pub fn new(side: Side, height: f32) -> Box<Object> {
        Box::new(Timber {
            side: side,
            transform: {
                let mut transform = Transform::IDENTITY;

                let x = match side {
                    Left => -2.0,
                    Right => 2.0,
                };

                transform.orientation.position = Vector::new(x, 0.0, 5.0);
                transform.orientation.rotation = Rotation::new(0.0, 0.0, 0.0);

                transform
            },
            mesh: {
                let vertices = vec![
                    Vector::new(1.0, height, -1.0),
                    Vector::new(-1.0, height, -1.0),
                    Vector::new(-1.0, 0.0, -1.0),
                    Vector::new(1.0, 0.0, -1.0),
                    Vector::new(1.0, 0.0, 1.0),
                    Vector::new(1.0, height, 1.0),
                    Vector::new(-1.0, height, 1.0),
                    Vector::new(-1.0, 0.0, 1.0),
                ];

                let normals = vertices.clone();

                let indices = vec![
                    0,1,2,  2,3,0,
                    0,3,4,  4,5,0,
                    0,5,6,  6,1,0,
                    1,6,7,  7,2,1,
                    7,4,3,  3,2,7,
                    4,7,6,  6,5,4,
                ];

                Mesh {
                    vertices: vertices,
                    normals: normals,
                    indices: indices,
                    color: Color::WHITE,
                    skew: Vector::new(0.1, 2.0, 0.1),
                    material: MaterialName::Wireframe,
                }
            },
            dead: false,
            height: height,
        })
    }
}

impl Object for Timber {
    fn transform_ref(&self) -> &Transform {
        &self.transform
    }

    fn transform_mut(&mut self) -> &mut Transform {
        &mut self.transform
    }

    fn meshes(&self) -> Vec<&Mesh> {
        vec![&self.mesh]
    }

    fn add(&self) -> Vec<Box<Object>> {
        if self.dead {
            use super::Flag;

            vec![Flag::new(self.side)]
        } else {
            Vec::with_capacity(0)
        }
    }

    fn remove(&self) -> bool {
        self.dead
    }

    fn update(&mut self, _: &(Joystick, Joystick)) {
        let offset = match self.side {
            Side::Left => -0.001,
            Side::Right => 0.001,
        };
        let z = self.transform.orientation.rotation.z + offset;

        let delta = unsafe { &::fun::global::TIMER }.delta;

        match self.side {
            Side::Left => {
                if z < -0.25 {
                    if unsafe { ::LEFT_PRAY } {
                        self.height -= 0.005 * delta;
                    }
                } else {
                    self.transform.orientation.rotation =  Rotation::new(0.0, 0.5, z);
                }
            },
            Side::Right => {
                if z > 0.25 {
                    if unsafe { ::RIGHT_PRAY } {
                        self.height -= 0.005 * delta;
                    }
                } else {
                    self.transform.orientation.rotation =  Rotation::new(0.0, 0.5, z);
                }
            },
        }

        self.mesh.skew.y = self.height;

        if self.height < 0.0 {
            self.dead = true;

            match self.side {
                Left => unsafe {
                    ::LEFT_FLAG_HEIGHT = 0.0;
                },
                Right => unsafe {
                    ::RIGHT_FLAG_HEIGHT = 0.0;
                },
            }
        }
    }
}
